defmodule Reseau.Etape.Parties do
  @moduledoc """
  Une étape pour proposer de rejoindre une partie ou créer une nouvelle partie.
  """

  @behaviour Reseau.Etape

  @typedoc "Un joueur"
  @type joueur() :: Reseau.Joueur.t()

  @typedoc "Une partie"
  @type partie() :: Reseau.Partie.t()

  @impl true
  @doc """
  Callback appelée quand l'étape est chargée et prête à être affichée.

  Arguments :

  * `joueur` : le joueur actuel (`Reseau.Joueur`) ;
  * `partie` : la partie actuelle (`Reseau.Partie`).

  Elle doit retourner un tuple contenant le joueur et la partie (éventuellement modifiés).

  """
  @spec entrer(joueur(), partie()) :: {joueur(), partie()}
  def entrer(joueur, partie) do
    parties =
      Jeu.lister_parties()
      |> Enum.map(fn {id, {processus, titre}} -> {id, processus, titre} end)

    joueur = %{joueur | mémoire: %{parties: parties}}
    {joueur, partie}
  end

  @impl true
  @doc """
  Retourne le titre de l'étape.
  """
  @spec titre(joueur(), partie()) :: String.t()
  def titre(_joueur, _partie), do: "Parties de jeux de cartes actuelles"

  @impl true
  @doc """
  Retourne le texte de l'étape.
  """
  @spec texte(joueur(), partie()) :: String.t()
  def texte(joueur, _partie) do
    titres =
      for {{_, _, titre}, index} <- Enum.with_index(joueur.mémoire.parties) do
        "#{index + 1} - #{titre}"
      end

    titres = Enum.join(titres, "\n")

    """
    Bienvenue, #{joueur.nom} !

    Pour jouer à une partie déjà créée, entrez simplement son numéro.
    Sinon, entrez C pour créer une nouvelle partie.

    Parties actuellement enregistrées :

    #{titres}
    """
  end

  @impl true
  @doc """
  Retourne le prompt de l'étape.
  """
  @spec prompt(joueur(), partie()) :: String.t()
  def prompt(_joueur, _partie), do: "Entrez le numéro de la partie à rejoindre ou C pour en créer une :"

  @impl true
  @doc """
  Gère les commandes entrées par l'utilisateur.

  La commande entrée est précisée sous la forme d'une chaîne de caractères. Le retour peut être de plusieurs types :

  - `:silence` : n'affiche rien et ne fait rien ;
  - `:prompt` : n'affiche rien sauf le prompt ;
  - `:rafraîchir` : n'affiche que l'écran de nouveau, sans message ;
  - `{:silence, message}` : affiche juste le message ;
  - `{:prompt, message}` : n'affiche que le message suivi du prompt ;
  - `{:rafraîchir, message}` : affiche le message suivi de tout l'écran ;
  - `{:rediriger, module, joueur, partie}` : change l'étape actuelle, affiche le nouvel écran ;
  - `{:rediriger, module, joueur, partie, message}` : change l'étape actuelle, affiche le message suivi du nouvel écran ;
  - `{:rediriger, module, joueur, partie, :silence}` : change l'étape actuelle, n'affiche rien ;
  - `{:rediriger, module, joueur, partie, :silence, message}` : change l'étape actuelle, affiche uniquement le message.

  """
  @spec gérer_commandes(joueur(), partie(), String.t()) ::
              :silence
              | :prompt
              | :rafrîchir
              | {:silence, String.t()}
              | {:prompt, String.t()}
              | {:rafraîchir, String.t()}
              | {:rediriger, module(), joueur(), partie()}
              | {:rediriger, module(), joueur(), partie(), String.t()}
              | {:rediriger, module(), joueur(), partie(), :silence}
              | {:rediriger, module(), joueur(), partie(), :silence, String.t()}
  def gérer_commandes(joueur, partie, "C") do
    règles = Jeu.Huit
    {id, processus} = Jeu.créer_partie(règles)
    {:ok, joueur_id} = Jeu.ajouter_joueur(id, processus, joueur.nom)
    joueur = %{joueur | id: joueur_id, mémoire: %{}}
    partie = %{partie | id: id, processus: processus, titre: règles.titre()}
    {:rediriger, Reseau.Etape.Plateau, joueur, partie}
  end

  def gérer_commandes(joueur, partie, commande) do
    with {index, _} <- Integer.parse(commande),
         {:ok, {id, processus, titre}} <- Enum.fetch(joueur.mémoire.parties, index - 1),
         {:ok, joueur_id} <- Jeu.ajouter_joueur(id, processus, joueur.nom) do
      joueur = %{joueur | id: joueur_id, mémoire: %{}}
      partie = %{partie | id: id, processus: processus, titre: titre}
      {:rediriger, Reseau.Etape.Plateau, joueur, partie}
    else
      _ -> {:prompt, "Cette partie n'est pas valide."}
    end
  end
end
